import { api } from "encore.dev/api";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

interface Karyawan {
  id: string;
  nip: string;
  nama_lengkap: string;
  tempat_lahir: string;
  tanggal_lahir: string;
  jenis_kelamin: string;
  agama: string;
  alamat: string;
  no_telepon: string;
  email: string;
  pendidikan_terakhir: string;
  jabatan: string;
  bagian: string;
  tanggal_masuk: string;
  status_kepegawaian: string;
}

interface ListKaryawanResponse {
  karyawan: Karyawan[];
  total: number;
}

interface ListKaryawanParams {
  madrasah_id?: string;
  search?: string;
  bagian?: string;
  limit?: number;
  offset?: number;
}

// Lists karyawan by madrasah
export const listKaryawan = api<ListKaryawanParams, ListKaryawanResponse>(
  { expose: true, method: "GET", path: "/karyawan" },
  async (params) => {
    const { madrasah_id, search = "", bagian, limit = 50, offset = 0 } = params;
    
    let whereClause = "";
    let searchParams: any[] = [];
    let paramIndex = 1;
    
    if (madrasah_id) {
      whereClause = `WHERE madrasah_id = $${paramIndex}`;
      searchParams.push(madrasah_id);
      paramIndex++;
    }
    
    if (search) {
      const searchCondition = `(nama_lengkap ILIKE $${paramIndex} OR nip ILIKE $${paramIndex} OR jabatan ILIKE $${paramIndex})`;
      whereClause = whereClause ? `${whereClause} AND ${searchCondition}` : `WHERE ${searchCondition}`;
      searchParams.push(`%${search}%`);
      paramIndex++;
    }
    
    if (bagian) {
      const bagianCondition = `bagian = $${paramIndex}`;
      whereClause = whereClause ? `${whereClause} AND ${bagianCondition}` : `WHERE ${bagianCondition}`;
      searchParams.push(bagian);
      paramIndex++;
    }
    
    const karyawanList = await db.rawQueryAll<Karyawan>(
      `SELECT id, nip, nama_lengkap, tempat_lahir, tanggal_lahir, jenis_kelamin, agama, alamat, 
              no_telepon, email, pendidikan_terakhir, jabatan, bagian, tanggal_masuk, status_kepegawaian
       FROM karyawan ${whereClause} 
       ORDER BY nama_lengkap 
       LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`,
      ...searchParams, limit, offset
    );
    
    const totalResult = await db.rawQueryRow<{ count: number }>(
      `SELECT COUNT(*) as count FROM karyawan ${whereClause}`,
      ...searchParams
    );
    
    return {
      karyawan: karyawanList,
      total: totalResult?.count || 0
    };
  }
);
