import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { useAuth } from "../contexts/AuthContext";
import Layout from "../components/Layout";
import ImportDialog from "../components/ImportDialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/components/ui/use-toast";
import { Plus, Search } from "lucide-react";
import backend from "~backend/client";
import type { CreateSiswaRequest } from "~backend/siswa/create_siswa";

export default function SiswaPage() {
  const { user } = useAuth();
  const [search, setSearch] = useState("");
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [formData, setFormData] = useState<CreateSiswaRequest>({
    madrasah_id: user?.madrasahId || "",
    nisn: "",
    nis: "",
    nama_lengkap: "",
    tempat_lahir: "",
    tanggal_lahir: "",
    jenis_kelamin: "",
    agama: "",
    alamat: "",
    no_telepon: "",
    nama_ayah: "",
    nama_ibu: "",
    pekerjaan_ayah: "",
    pekerjaan_ibu: "",
    kelas: "",
    tahun_masuk: new Date().getFullYear(),
    status: "aktif",
  });

  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data, isLoading } = useQuery({
    queryKey: ["siswa", user?.madrasahId, search],
    queryFn: () => backend.siswa.listSiswa({ 
      madrasah_id: user?.userType !== "superadmin" ? user?.madrasahId : undefined,
      search, 
      limit: 100 
    }),
  });

  const createMutation = useMutation({
    mutationFn: (data: CreateSiswaRequest) => backend.siswa.createSiswa(data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["siswa"] });
      setIsDialogOpen(false);
      setFormData({
        madrasah_id: user?.madrasahId || "",
        nisn: "",
        nis: "",
        nama_lengkap: "",
        tempat_lahir: "",
        tanggal_lahir: "",
        jenis_kelamin: "",
        agama: "",
        alamat: "",
        no_telepon: "",
        nama_ayah: "",
        nama_ibu: "",
        pekerjaan_ayah: "",
        pekerjaan_ibu: "",
        kelas: "",
        tahun_masuk: new Date().getFullYear(),
        status: "aktif",
      });
      toast({
        title: "Berhasil",
        description: "Data siswa berhasil ditambahkan",
      });
    },
    onError: (error) => {
      console.error("Create siswa error:", error);
      toast({
        title: "Gagal",
        description: "Gagal menambahkan data siswa",
        variant: "destructive",
      });
    },
  });

  const importMutation = useMutation({
    mutationFn: (data: any[]) => {
      console.log("Import siswa data received:", data);
      
      const mappedData = data.map((row, index) => {
        console.log(`Processing siswa row ${index + 1}:`, row);
        
        return {
          nisn: String(row["NISN"] || row["nisn"] || "").trim(),
          nis: String(row["NIS"] || row["nis"] || "").trim(),
          nama_lengkap: String(row["Nama Lengkap"] || row["nama_lengkap"] || "").trim(),
          tempat_lahir: String(row["Tempat Lahir"] || row["tempat_lahir"] || "").trim(),
          tanggal_lahir: String(row["Tanggal Lahir"] || row["tanggal_lahir"] || "").trim(),
          jenis_kelamin: String(row["Jenis Kelamin"] || row["jenis_kelamin"] || "").trim(),
          agama: String(row["Agama"] || row["agama"] || "").trim(),
          alamat: String(row["Alamat"] || row["alamat"] || "").trim(),
          no_telepon: String(row["No Telepon"] || row["no_telepon"] || "").trim(),
          nama_ayah: String(row["Nama Ayah"] || row["nama_ayah"] || "").trim(),
          nama_ibu: String(row["Nama Ibu"] || row["nama_ibu"] || "").trim(),
          pekerjaan_ayah: String(row["Pekerjaan Ayah"] || row["pekerjaan_ayah"] || "").trim(),
          pekerjaan_ibu: String(row["Pekerjaan Ibu"] || row["pekerjaan_ibu"] || "").trim(),
          kelas: String(row["Kelas"] || row["kelas"] || "").trim(),
          tahun_masuk: parseInt(String(row["Tahun Masuk"] || row["tahun_masuk"] || new Date().getFullYear())),
          status: String(row["Status"] || row["status"] || "aktif").trim(),
        };
      });
      
      console.log("Mapped siswa data:", mappedData);
      return backend.siswa.importSiswa({
        madrasah_id: user?.madrasahId || "",
        data: mappedData
      });
    },
    onSuccess: (result) => {
      queryClient.invalidateQueries({ queryKey: ["siswa"] });
      toast({
        title: "Import Selesai",
        description: `Berhasil: ${result.success}, Gagal: ${result.failed}`,
      });
      if (result.errors.length > 0) {
        console.error("Import errors:", result.errors);
        toast({
          title: "Import Warnings",
          description: `Ada ${result.failed} data yang gagal diimport. Lihat console untuk detail.`,
          variant: "destructive",
        });
      }
    },
    onError: (error) => {
      console.error("Import siswa error:", error);
      toast({
        title: "Import Gagal",
        description: "Gagal mengimport data siswa",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    createMutation.mutate(formData);
  };

  const handleImport = async (data: any[]) => {
    console.log("handleImport siswa called with:", data);
    await importMutation.mutateAsync(data);
  };

  const templateFields = [
    "NISN",
    "NIS",
    "Nama Lengkap",
    "Tempat Lahir",
    "Tanggal Lahir",
    "Jenis Kelamin",
    "Agama",
    "Alamat",
    "No Telepon",
    "Nama Ayah",
    "Nama Ibu",
    "Pekerjaan Ayah",
    "Pekerjaan Ibu",
    "Kelas",
    "Tahun Masuk",
    "Status"
  ];

  return (
    <Layout>
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Data Siswa</h1>
            <p className="text-gray-600 mt-2">Kelola data siswa di madrasah</p>
          </div>
          <div className="flex space-x-2">
            <ImportDialog
              title="Siswa"
              templateFields={templateFields}
              onImport={handleImport}
              isLoading={importMutation.isPending}
            />
            <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
              <DialogTrigger asChild>
                <Button className="bg-teal-600 hover:bg-teal-700">
                  <Plus className="h-4 w-4 mr-2" />
                  Tambah Siswa
                </Button>
              </DialogTrigger>
              <DialogContent className="max-w-2xl">
                <DialogHeader>
                  <DialogTitle>Tambah Siswa Baru</DialogTitle>
                </DialogHeader>
                <form onSubmit={handleSubmit} className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="nisn">NISN</Label>
                      <Input
                        id="nisn"
                        value={formData.nisn}
                        onChange={(e) => setFormData({ ...formData, nisn: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="nis">NIS</Label>
                      <Input
                        id="nis"
                        value={formData.nis}
                        onChange={(e) => setFormData({ ...formData, nis: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="nama_lengkap">Nama Lengkap *</Label>
                    <Input
                      id="nama_lengkap"
                      value={formData.nama_lengkap}
                      onChange={(e) => setFormData({ ...formData, nama_lengkap: e.target.value })}
                      required
                    />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="tempat_lahir">Tempat Lahir</Label>
                      <Input
                        id="tempat_lahir"
                        value={formData.tempat_lahir}
                        onChange={(e) => setFormData({ ...formData, tempat_lahir: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="tanggal_lahir">Tanggal Lahir</Label>
                      <Input
                        id="tanggal_lahir"
                        type="date"
                        value={formData.tanggal_lahir}
                        onChange={(e) => setFormData({ ...formData, tanggal_lahir: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="jenis_kelamin">Jenis Kelamin</Label>
                      <Input
                        id="jenis_kelamin"
                        value={formData.jenis_kelamin}
                        onChange={(e) => setFormData({ ...formData, jenis_kelamin: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="agama">Agama</Label>
                      <Input
                        id="agama"
                        value={formData.agama}
                        onChange={(e) => setFormData({ ...formData, agama: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="alamat">Alamat</Label>
                    <Input
                      id="alamat"
                      value={formData.alamat}
                      onChange={(e) => setFormData({ ...formData, alamat: e.target.value })}
                    />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="nama_ayah">Nama Ayah</Label>
                      <Input
                        id="nama_ayah"
                        value={formData.nama_ayah}
                        onChange={(e) => setFormData({ ...formData, nama_ayah: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="nama_ibu">Nama Ibu</Label>
                      <Input
                        id="nama_ibu"
                        value={formData.nama_ibu}
                        onChange={(e) => setFormData({ ...formData, nama_ibu: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="pekerjaan_ayah">Pekerjaan Ayah</Label>
                      <Input
                        id="pekerjaan_ayah"
                        value={formData.pekerjaan_ayah}
                        onChange={(e) => setFormData({ ...formData, pekerjaan_ayah: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="pekerjaan_ibu">Pekerjaan Ibu</Label>
                      <Input
                        id="pekerjaan_ibu"
                        value={formData.pekerjaan_ibu}
                        onChange={(e) => setFormData({ ...formData, pekerjaan_ibu: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="grid grid-cols-3 gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="kelas">Kelas</Label>
                      <Input
                        id="kelas"
                        value={formData.kelas}
                        onChange={(e) => setFormData({ ...formData, kelas: e.target.value })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="tahun_masuk">Tahun Masuk</Label>
                      <Input
                        id="tahun_masuk"
                        type="number"
                        value={formData.tahun_masuk}
                        onChange={(e) => setFormData({ ...formData, tahun_masuk: parseInt(e.target.value) })}
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="no_telepon">No. Telepon</Label>
                      <Input
                        id="no_telepon"
                        value={formData.no_telepon}
                        onChange={(e) => setFormData({ ...formData, no_telepon: e.target.value })}
                      />
                    </div>
                  </div>
                  <div className="flex justify-end space-x-2">
                    <Button
                      type="button"
                      variant="outline"
                      onClick={() => setIsDialogOpen(false)}
                    >
                      Batal
                    </Button>
                    <Button
                      type="submit"
                      className="bg-teal-600 hover:bg-teal-700"
                      disabled={createMutation.isPending}
                    >
                      {createMutation.isPending ? "Menyimpan..." : "Simpan"}
                    </Button>
                  </div>
                </form>
              </DialogContent>
            </Dialog>
          </div>
        </div>

        <Card>
          <CardHeader>
            <div className="flex items-center space-x-2">
              <Search className="h-4 w-4 text-gray-400" />
              <Input
                placeholder="Cari siswa..."
                value={search}
                onChange={(e) => setSearch(e.target.value)}
                className="max-w-sm"
              />
            </div>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="text-center py-8">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-teal-600 mx-auto"></div>
                <p className="text-gray-600 mt-2">Memuat data...</p>
              </div>
            ) : (
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead>
                    <tr className="border-b">
                      <th className="text-left py-3 px-4 font-medium text-gray-900">NISN</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Nama Lengkap</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Kelas</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Tahun Masuk</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">Status</th>
                      <th className="text-left py-3 px-4 font-medium text-gray-900">No. Telepon</th>
                    </tr>
                  </thead>
                  <tbody>
                    {data?.siswa.map((siswa) => (
                      <tr key={siswa.id} className="border-b hover:bg-gray-50">
                        <td className="py-3 px-4 text-sm">{siswa.nisn}</td>
                        <td className="py-3 px-4 text-sm font-medium">{siswa.nama_lengkap}</td>
                        <td className="py-3 px-4 text-sm">{siswa.kelas}</td>
                        <td className="py-3 px-4 text-sm">{siswa.tahun_masuk}</td>
                        <td className="py-3 px-4 text-sm">
                          <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                            siswa.status === 'aktif' ? 'bg-green-100 text-green-800' :
                            'bg-gray-100 text-gray-800'
                          }`}>
                            {siswa.status}
                          </span>
                        </td>
                        <td className="py-3 px-4 text-sm">{siswa.no_telepon}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
                {data?.siswa.length === 0 && (
                  <div className="text-center py-8 text-gray-600">
                    Tidak ada data siswa
                  </div>
                )}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </Layout>
  );
}
