import { api } from "encore.dev/api";
import { db } from "./db";

interface Madrasah {
  id: string;
  nsm: string;
  npsn: string;
  nama_madrasah: string;
  alamat: string;
  kecamatan: string;
  kabupaten: string;
  provinsi: string;
  akreditasi: string;
  status: string;
  nama_kepala: string;
  operator_token?: string;
  operator_registered?: boolean;
}

interface ListMadrasahResponse {
  madrasah: Madrasah[];
  total: number;
}

interface ListMadrasahParams {
  search?: string;
  limit?: number;
  offset?: number;
}

// Lists all madrasah (for superadmin)
export const listMadrasah = api<ListMadrasahParams, ListMadrasahResponse>(
  { expose: true, method: "GET", path: "/madrasah" },
  async (params) => {
    const { search = "", limit = 50, offset = 0 } = params;
    
    let whereClause = "";
    let searchParams: any[] = [];
    
    if (search) {
      whereClause = "WHERE nama_madrasah ILIKE $1 OR nsm ILIKE $1 OR kecamatan ILIKE $1";
      searchParams = [`%${search}%`];
    }
    
    const madrasahList = await db.rawQueryAll<Madrasah>(
      `SELECT id, nsm, npsn, nama_madrasah, alamat, kecamatan, kabupaten, provinsi, akreditasi, status, nama_kepala,
              operator_token, operator_registered
       FROM madrasah ${whereClause} 
       ORDER BY nama_madrasah 
       LIMIT $${searchParams.length + 1} OFFSET $${searchParams.length + 2}`,
      ...searchParams, limit, offset
    );
    
    const totalResult = await db.rawQueryRow<{ count: number }>(
      `SELECT COUNT(*) as count FROM madrasah ${whereClause}`,
      ...searchParams
    );
    
    return {
      madrasah: madrasahList,
      total: totalResult?.count || 0
    };
  }
);
